/*
 * $Id: //devel/tools/main/datemath/util.c#1 $
 *
 * written by:  Stephen J. Friedl
 *              Software Consultant
 *              Tustin, California USA
 *              steve@unixwiz.net / www.unixwiz.net
 *
 *	These are various utility functions that are used throughout
 *	the datemath program.
 */
#include <stdio.h>
#include <assert.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "defs.h"

int	century19_year = 40;	/* 0..39 = 2000 / 40..99 = 1999 */

/*
 * jultoyymm()
 *
 *	Convert the Julian date to YYMM.  Note that if the jdate
 *	provided is < 1900, we treat it as 1900.  Return is -1 on
 *	error.
 */
int jultoyymm(jdate_t jdate)
{
short	mdy[3];

	if (rjulmdy(jdate, mdy) < 0)		/* convert to mdy[]	*/
		return(-1);

	if (mdy[YY] < 1900)			/* too early?		*/
		mdy[YY] = 1900;

	mdy[YY] %= 100;				/* strip off 19xx	*/

	return (mdy[YY]*100) + mdy[MM];
}

/*
 * yymmtojul.c
 *
 *	This converts a YYMM date to a Julian date, which is returned.
 *	The "eflag" indicates whether the date returned should be the
 *	end of the month or the beginning.  This assumes that YY is in
 *	*this* century, and it dies on error.
 */
jdate_t yymmtojul(int yymm, int eflag)
{
short	mdy[3];
jdate_t	jdate;
int	rv;

	mdy[MM] = yymm % 100;
	mdy[YY] = yymm / 100;
	mdy[DD] = eflag ? daysinmm(mdy[MM], mdy[YY]) : 1;

	mdy[YY] += 1900;

	if (rv = rmdyjul(mdy, &jdate), rv < 0)
		die("bad date conversion in yymmtojul-(%d)", rv);
	else
		return(jdate);
}

/*
 * yymm_add()
 *
 *	Given a YYMM date (which must be valid), add the given
 *	number of months to it.  The number of months may be
 *	negative, and the new YYMM is returned.
 */
int yymm_add(int yymm, int nmonths)
{
register int	mm = yymm % 100,	/* month of the guy	*/
		yy = yymm / 100;	/* year of the guy	*/

	mm += nmonths;

	while (mm < 1)			/* year got smaller?	*/
		mm += 12, yy--;

	while (mm > 12)			/* year got bigger?	*/
		mm -= 12, yy++;

	return(yy * 100 + mm);
}


/*
 * die()
 *
 *	Given a format and some args, print an error message and
 *	exit.  The program depends on the external variable ProgName,
 *	which should be set to argv[0].  The format string should not
 *	contain a newline, as one is appended for you.  All output
 *	goes to stderr.
 */

void die(const char *format, ...)
{
va_list	args;

	va_start(args, format);

	if (ProgName)
		(void)fprintf(stderr, "%s: ", ProgName);

	vfprintf(stderr, format, args);
	fputc('\n', stderr);

	va_end(args);

	exit(EXIT_FAILURE);
}

/*
 * year_to_yyyy()
 *
 *	Given a year that could be in two-digit or four-digit
 *	format, figure out a four-digit year for it. We have a
 *	variable cutoff for 1900/2000.
 */
int year_to_yyyy(int year)
{
	if ( year < century19_year )
		year += 2000;

	else if ( year < 100 )
		year += 1900;

	return year;
}

/*
 * daysinmm()
 *
 *	Given a month (1-based) and a year, return the number
 *	of days in that month.  The year may be either YY or YYYY
 *	with the full prefix.  We take care to properly calculate
 *	the number of days in February according to the following
 *	rule:
 *
 *		if the year is divisible by for, except years
 *		divisible by 100, but including years divisible
 *		by 400.
 *
 *	If there is no leading YY, 1900 is assumed.
 */

int daysinmm(int mm, int yy)
{
static int mdays[12] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

	if (mm < 1 || mm > 12)			/* bogus dates?		*/
		return(-1);

	if (mm != 2)				/* not February?	*/
		return(mdays[mm-1]);

	yy = year_to_yyyy(yy);

#define		YYMOD(n)	((yy % (n)) == 0)

	return(28 + (YYMOD(4) && (YYMOD(400) || !YYMOD(100))));
}

/*
 * daysinyymm.c
 *
 *	Given a YYMM date, return the number of days in that month.
 */
int daysinyymm(int yymm)
{
	return(daysinmm(yymm % 100, yymm / 100));
}

/*
 * strlower()
 *
 *	Given a string, convert it to all lower case.
 */
char *strlower(char *s)
{
char	*s_save = s;

	assert(s != 0);

	for (; *s; s++)
	{
		if (isupper(*s))
			*s = tolower(*s);
	}
	return s_save;
}
